//
//      pmap_common.tcc - common template code for priority_map,
//                        priority_multimap, indexed_priority_map,
//                        indexed_priority_multimap
//
//      VERSION 1.0.0
//
//      Copyright (c) 2003,2005,2006,2007 Pittsburgh Supercomputing Center
//
//      This program is free software; you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation; either version 2 of the License, or
//      (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.  Neither the
//      Pittsburgh Supercomputing Center nor any of the authors assume
//      any liability for damages, incidental or otherwise, caused by the
//      installation or use of this software.
//
//      You should have received a copy of the GNU General Public License
//      along with this program; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin Street, Fifth Floor,
//      Boston, MA  02110-1301  USA
//
//
//      This implementation is based on the splay tree code
//      (top-down-size-splay.c) written by Daniel Sleator, and released
//      into the public domain.
//
//	HISTORY
//      	3/03	Written by Greg Hood, PSC
//              3/05    Removed NEOSIM dependencies (ghood@psc.edu)
//              10/05   Extended from indexed_priority_map (ghood@psc.edu)
//              1/07    Fixed compilation error in erase(iterator&,iterator&)
//                        (ghood@psc.edu)
//

#undef _PMAP_TEMPLATE_DEF
#undef _PMAP_TEMPLATE
#undef _PMAP_ITEM
#ifdef _PMAP_INDEXED
  #define _PMAP_TEMPLATE_DEF	template<class Key, class Value, class Index, class H, class L, class A>
  #define _PMAP_TEMPLATE	_PMAP_CLASS<Key,Value,Index,H,L,A>
  #define _PMAP_ITEM		triple<Key,Value,Index>
#else
  #define _PMAP_TEMPLATE_DEF	template<class Key, class Value, class H, class L, class A>
  #define _PMAP_TEMPLATE	_PMAP_CLASS<Key,Value,H,L,A>
  #define _PMAP_ITEM		pair<Key,Value>
#endif

#undef _PMAP_METHOD
#ifdef _PMAP_HAVE_EXPORT
  #define _PMAP_METHOD		export _PMAP_TEMPLATE_DEF
#else
  #define _PMAP_METHOD		_PMAP_TEMPLATE_DEF
#endif

#undef _PMAP_STRING
#undef _PMAP_ERROR
#define _PMAP_STRING(s)		#s
#define _PMAP_ERROR(x)		throw(error(_PMAP_STRING(_PMAP_CLASS) ": " x))

namespace pmap
{
  using std::size_t;
  using std::pair;
  using std::ptrdiff_t;
  using std::vector;

  _PMAP_METHOD
  _PMAP_TEMPLATE::_PMAP_CLASS (const H& h, const L& l, const A& a) :
    hashFunctor(h), lessFunctor(l), hashTable(a), table(a)
  {
    logSize = 0;
    root = 0;
    nElements = 0;
    freeList = 0;
    Entry e;
#ifdef _PMAP_INDEXED
    e.item.third = Index(0);
    e.rank = e.item.third;
#endif
    table.push_back(e);
    hashTable.resize(1);
  };

  _PMAP_METHOD
  _PMAP_TEMPLATE::_PMAP_CLASS (const _PMAP_CLASS& x) :
    logSize(x.logSize),
    freeList(x.freeList),
    nElements(x.nElements),
    root(x.root),
    hashTable(x.hashTable),
    table(x.table)
  {
  }

  _PMAP_METHOD
  _PMAP_TEMPLATE& _PMAP_TEMPLATE::operator= (const _PMAP_CLASS& x)
  {
    logSize = x.logSize;
    freeList = x.freeList;
    nElements = x.nElements;
    root = x.root;
    hashTable = x.hashTable;
    table = x.table;
    return(*this);
  }

  _PMAP_METHOD
  void _PMAP_TEMPLATE::swap (_PMAP_CLASS& x)
  {
    std::swap(logSize, x.logSize);
    std::swap(freeList, x.freeList);
    std::swap(nElements, x.nElements);
    std::swap(root, x.root);
    hashTable.swap(x.hashTable);
    table.swap(x.table);
  }

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::iterator&
  _PMAP_TEMPLATE::iterator::operator++ ()
  {
    const vector<Entry>& table(ipm->table);
    if (index == 0)
      return(*this);
    if (table[index].right != 0)
      {
	index = table[index].right;
	while (table[index].left != 0)
	  index = table[index].left;
      }
    else
      for (;;)
	{
	  size_type parent = table[index].parent;
	  if (parent == 0)
	    {
	      index = 0;
	      break;
	    }
	  else if (table[parent].left == index)
	    {
	      index = parent;
	      break;
	    }
	  else if (table[parent].right == index)
	    index = parent;
	  else
	    _PMAP_ERROR("corrupt tree (operator++)");
	}
    return(*this);
  }

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::iterator
  _PMAP_TEMPLATE::iterator::operator++ (int)
  {
    typename _PMAP_TEMPLATE::iterator iter = *this;
    ++*this;
    return(iter);
  }

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::iterator&
  _PMAP_TEMPLATE::iterator::operator-- ()
  {
    const vector<Entry>& table(ipm->table);
    if (index == 0)
      {
	index = ipm->root;
	if (index == 0)
	  return(*this);
	while (table[index].right != 0)
	  index = table[index].right;
      }
    else if (table[index].left != 0)
      {
	index = table[index].left;
	while (table[index].right != 0)
	  index = table[index].right;
      }
    else
      for (;;)
	{
	  size_type parent = table[index].parent;
	  if (parent == 0)
	    {
	      index = 0;
	      break;
	    }
	  else if (table[parent].right == index)
	    {
	      index = parent;
	      break;
	    }
	  else if (table[parent].left == index)
	    index = parent;
	  else
	    _PMAP_ERROR("corrupt tree (operator--)");
	}
    return(*this);
  }

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::iterator
  _PMAP_TEMPLATE::iterator::operator-- (int)
  {
    typename _PMAP_TEMPLATE::iterator iter = *this;
    --*this;
    return(iter);
  }

  _PMAP_METHOD
  bool
  _PMAP_TEMPLATE::iterator::operator< (const iterator& x) const
  {
    if (ipm != x.ipm)
      _PMAP_ERROR("ordering operator (<,>, <=, >=) applied to iterators not in same container");
    if (index == x.index)
      return(false);

    const vector<Entry>& table(ipm->table);
    if (ipm->lessFunctor(table[index].item.second,
			 table[x.index].item.second))
      return(true);
    if (ipm->lessFunctor(table[x.index].item.second,
			 table[index].item.second))
      return(false);

    vector<size_type> parents0, parents1;

    size_type i = index;
    do {
      parents0.push_back(i);
      i = table[i].parent;
    } while (i != 0);
    i = x.index;
    do {
      parents1.push_back(i);
      i = table[i].parent;
    } while (i != 0);

    for (i = 1; i < parents0.size() && i < parents1.size(); ++i)
      if (parents0[i] != parents1[i])
	{
	  if (table[parents0[i-1]].left == parents0[i] &&
	      table[parents0[i-1]].right == parents1[i])
	    return(true);
	  if (table[parents0[i-1]].left == parents1[i] &&
	      table[parents0[i-1]].right == parents0[i])
	    return(false);
	  _PMAP_ERROR("internal error -- corrupt tree structure");
	}
    if (i >= parents0.size())
      {
	if (i >= parents1.size())
	  _PMAP_ERROR("internal error -- corrupt tree structure");
	if (table[parents0[i-1]].left == parents1[i])
	  return(false);
	if (table[parents0[i-1]].right == parents1[i])
	  return(true);
	_PMAP_ERROR("internal error -- corrupt tree structure");
      }
    else
      {
	if (table[parents0[i-1]].left == parents0[i])
	  return(true);
	if (table[parents0[i-1]].right == parents0[i])
	  return(false);
	_PMAP_ERROR("internal error -- corrupt tree structure");
      }
  }

  _PMAP_METHOD
  bool
  _PMAP_TEMPLATE::iterator::operator> (const iterator& x) const
  {
    return((*this != x) && !(*this < x));
  }

  _PMAP_METHOD
  bool
  _PMAP_TEMPLATE::iterator::operator<= (const iterator& x) const
  {
    return((*this == x) || (*this < x));
  }

  _PMAP_METHOD
  bool
  _PMAP_TEMPLATE::iterator::operator>= (const iterator& x) const
  {
    return(!(*this < x));
  }


#ifdef _PMAP_INDEXED

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::iterator
  _PMAP_TEMPLATE::iterator::operator+ (const Index x)
  {
    Index idx = ipm->index(*this) + x;
    return(ipm->index_find(idx));
  }

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::iterator&
  _PMAP_TEMPLATE::iterator::operator+= (const Index x)
  {
    *this = *this + x;
  }

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::iterator
  _PMAP_TEMPLATE::iterator::operator- (const Index x)
  {
    Index idx = ipm->index(*this) - x;
    return(ipm->index_find(idx));
  }

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::iterator&
  _PMAP_TEMPLATE::iterator::operator-= (const Index x)
  {
    *this = *this - x;
  }

  _PMAP_METHOD
  const _PMAP_ITEM&
  _PMAP_TEMPLATE::iterator::operator[] (const Index x)
  {
    iterator i = ipm->index_find(x);
    if (i == ipm->end())
      _PMAP_ERROR("index out-of-bounds (operator[])");
    return(*i);
  }

#endif

  _PMAP_METHOD
  void _PMAP_TEMPLATE::insertInTree (size_type index)
  {
    if (root == 0)
      table[index].left = table[index].right = 0;
    else
      {
#ifdef _PMAP_DEBUG
	if (root != 0)
	  checkSubtree(root, 0,
		       false, table[root].item.second,
		       false, table[root].item.second);
#endif
	splay(index);
#ifdef _PMAP_DEBUG
	if (root != 0)
	  checkSubtree(root, 0,
		       false, table[root].item.second,
		       false, table[root].item.second);
#endif
	if (lessFunctor(table[index].item.second, table[root].item.second) ||
	    !lessFunctor(table[root].item.second, table[index].item.second) &&
	    index < root)
	  {
	    table[index].left = table[root].left;
	    table[table[index].left].parent = index;
	    table[index].right = root;
	    table[table[index].right].parent = index;
	    table[root].left = 0;
#ifdef _PMAP_INDEXED
	    table[root].rank = table[root].item.third +
	      table[table[root].right].rank;
#endif
	  }
	else
	  {
	    table[index].right = table[root].right;
	    table[table[index].right].parent = index;
	    table[index].left = root;
	    table[table[index].left].parent = index;
	    table[root].right = 0;
#ifdef _PMAP_INDEXED
	    table[root].rank = table[root].item.third +
	      table[table[root].left].rank;
#endif
	  }
      }
#ifdef _PMAP_INDEXED
    table[index].rank = table[index].item.third +
      table[table[index].left].rank +
      table[table[index].right].rank;
#endif
    root = index;
    table[root].parent = 0;
  }

  _PMAP_METHOD
  void _PMAP_TEMPLATE::removeFromTree (size_type index)
  {
#ifdef _PMAP_DEBUG
    if (root != 0)
      checkSubtree(root, 0,
		   false, table[root].item.second,
		   false, table[root].item.second);
#endif
    splay(index);
    if (root != index)
      _PMAP_ERROR("splay did not bring index to root");
#ifdef _PMAP_DEBUG
    if (root != 0)
      checkSubtree(root, 0,
		   false, table[root].item.second,
		   false, table[root].item.second);
#endif

    if (table[root].left == 0)
      {
	root = table[root].right;
	table[root].parent = 0;
#ifdef _PMAP_DEBUG
	if (root != 0)
	  checkSubtree(root, 0,
		       false, table[root].item.second,
		       false, table[root].item.second);
#endif
      }
    else if (table[root].right == 0)
      {
	root = table[root].left;
	table[root].parent = 0;
#ifdef _PMAP_DEBUG
	if (root != 0)
	  checkSubtree(root, 0,
		       false, table[root].item.second,
		       false, table[root].item.second);
#endif
      }
    else
      {
	size_type r = table[root].right;
	root = table[root].left;
	table[root].parent = 0;
	size_type x = root;
	while (table[x].right != 0)
	  x = table[x].right;
	splay(x);
	if (root != x)
	  _PMAP_ERROR("splay did not bring index to root");
	table[root].right = r;
	table[r].parent = root;
#ifdef _PMAP_INDEXED
	table[root].rank = table[root].rank + table[r].rank;
#endif
#ifdef _PMAP_DEBUG
	if (root != 0)
	  checkSubtree(root, 0,
		       false, table[root].item.second,
		       false, table[root].item.second);
#endif
      }
#ifdef _PMAP_DEBUG
    if (root != 0)
      checkNotInTree(root, index);
#endif
  }

  _PMAP_METHOD
#ifdef _PMAP_MULTI
  typename _PMAP_TEMPLATE::iterator
#else
  pair<typename _PMAP_TEMPLATE::iterator,bool>
#endif
  _PMAP_TEMPLATE::insert (const Key& k,
			  const Value& v
#ifdef _PMAP_INDEXED
			  , const Index& delta
#endif
			  )
  {
#ifdef _PMAP_DEBUG
    check();
#endif

    // try to find key
    size_type hv = hashFunctor(k) % hashTable.size();
    size_type index;
#ifndef _PMAP_MULTI
    index = hashTable[hv];
    while (index > 0 && !(table[index].item.first == k))
      index = table[index].next;
    if (index == 0)
      {
#endif
	// test if the entry table is full
	while ((index = freeList) == 0)
	  {
	    enlarge();
	    hv = hashFunctor(k) % hashTable.size();
	  }
	freeList = table[index].next;
      
	// make a new entry
	table[index].item.first = k;
	++nElements;
      
	// put in hash table
	table[index].next = hashTable[hv];
	hashTable[hv] = index;
#ifdef _PMAP_MULTI
	table[table[index].next].prev = index;
	table[index].prev = 0;
#endif
#ifdef _PMAP_DEBUG
	if (root != 0)
	  checkSubtree(root, 0,
		       false, table[root].item.second,
		       false, table[root].item.second);
#endif
#ifndef _PMAP_MULTI
      }
    else
      return(pair<iterator,bool>(iterator(this, index), false));
#endif

#ifdef _PMAP_INDEXED
    table[index].item.third = delta;
    table[index].rank = table[index].item.third;
#endif
    table[index].item.second = v;
    table[index].left = 0;
    table[index].right = 0;
    table[index].parent = 0;
#ifdef _PMAP_DEBUG
    if (root != 0)
      checkSubtree(root, 0,
		   false, table[root].item.second,
		   false, table[root].item.second);
#endif
  
    // update the tree
    insertInTree(index);

#ifdef _PMAP_DEBUG
    check();
#endif
#ifdef _PMAP_MULTI
    return(iterator(this, index));
#else
    return(pair<iterator,bool>(iterator(this, index), true));
#endif
  }

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::size_type
  _PMAP_TEMPLATE::erase (const Key& k)
  {
#ifdef _PMAP_DEBUG
    check();
#endif

    // try to find key
    size_type hv = hashFunctor(k) % hashTable.size();
    size_type index = hashTable[hv];
    size_type prev = 0;
#ifdef _PMAP_MULTI
    size_type next;
    size_type count = 0;
    while (index != 0)
      {
	next = table[index].next;
	if (table[index].item.first == k)
	  {
	    // remove from hash table
	    if (prev != 0)
	      table[prev].next = table[index].next;
	    else
	      hashTable[hv] = table[index].next;
	    table[next].prev = prev;
	    removeFromTree(index);
	    table[index].next = freeList;
	    freeList = index;
	    --nElements;
	    ++count;
	  }
	else
	  prev = index;
	index = next;
      }
#ifdef _PMAP_DEBUG
    check();
#endif
    return(count);
#else
    while (index != 0 && !(table[index].item.first == k))
      {
	prev = index;
	index = table[index].next;
      }
    if (index == 0)
      return(0);
    // remove from hash table
    if (prev != 0)
      table[prev].next = table[index].next;
    else
      hashTable[hv] = table[index].next;
    removeFromTree(index);
    table[index].next = freeList;
    freeList = index;
    --nElements;
#ifdef _PMAP_DEBUG
    check();
#endif
    return(1);
#endif
  }

  _PMAP_METHOD
  void _PMAP_TEMPLATE::erase (iterator& pos)
  {
#ifdef _PMAP_DEBUG
    check();
#endif

    if (pos.ipm != this)
      _PMAP_ERROR("erase() of iterator not in container");

    if (pos.index == 0)
      return;
  
    // prepare to update iterator to the next position
    iterator npos = pos;
    ++npos;

    // find entry
    size_type prev = 0;
#ifdef _PMAP_MULTI
    // remove from hash table
    prev = table[pos.index].prev;
    if (prev == 0)
      {
	size_type hv = hashFunctor(table[pos.index].item.first) %
	  hashTable.size();
	size_type index = hashTable[hv];
	if (index == 0)
	  _PMAP_ERROR("internal error: key not found in hash table");
	hashTable[hv] = table[index].next;
      }
    else
      table[prev].next = table[pos.index].next;
    table[table[pos.index].next].prev = prev;
#else
    size_type hv = hashFunctor(table[pos.index].item.first) % hashTable.size();
    size_type index = hashTable[hv];
    while (index != 0 && index != pos.index)
      {
	prev = index;
	index = table[index].next;
      }
    if (index == 0)
      _PMAP_ERROR("internal error: key not found in hash table");
    
    // remove from hash table
    if (prev != 0)
      table[prev].next = table[index].next;
    else
      hashTable[hv] = table[index].next;
#endif

    removeFromTree(pos.index);

    table[pos.index].next = freeList;
    freeList = pos.index;
    --nElements;
    pos = npos;

#ifdef _PMAP_DEBUG
    check();
#endif
  }

  _PMAP_METHOD
  void _PMAP_TEMPLATE::erase (iterator& first,
			      iterator& last)
  {
#ifdef _PMAP_DEBUG
    check();
#endif

    if (first.ipm != this || last.ipm != this)
      _PMAP_ERROR("erase() of iterator not in container");

    if (first > last)
      return;

    iterator i = first;
    iterator ni;
    bool done = false;
    do
      {
	ni = i;
	++ni;
	if (i == last)
	  done = true;
	erase(i);
	i = ni;
      } while (!done);

#ifdef _PMAP_DEBUG
    check();
#endif
  }

#ifdef _PMAP_INDEXED
  _PMAP_METHOD
  Index _PMAP_TEMPLATE::index (const Key& k) const
  {
    // look up key
    size_type hv = hashFunctor(k) % hashTable.size();
    size_type index = hashTable[hv];
    while (index > 0 && !(table[index].item.first == k))
      index = table[index].next;
    if (index == 0)
      return(this->index(end()));
  
    // go up the tree
    Index rank = table[table[index].left].rank;
    size_type parent;
    while (table[index].parent != 0)
      {
	parent = table[index].parent;
	if (table[parent].left == index)
	  {
	    // we are the left child; we exclude the parent node
	  }
	else if (table[parent].right == index)
	  {
	    // we are the right child; we include the parent node plus the
	    //   left subtree
	    rank = rank + table[table[parent].left].rank +
	      table[parent].item.third;
	  }
	else
	  _PMAP_ERROR("corrupt tree (index)");
	index = parent;
      }
    return(rank);
  }

  _PMAP_METHOD
  Index _PMAP_TEMPLATE::index (const iterator& i) const
  {
    if (i.ipm != this)
      _PMAP_ERROR("index() of iterator not in container");
    size_type index = i.index;
    if (index == 0)
      return(table[root].rank);
  
    // go up the tree
    Index rank = table[table[index].left].rank;
    size_type parent;
    while (table[index].parent != 0)
      {
	parent = table[index].parent;
	if (table[parent].left == index)
	  {
	    // we are the left child; we exclude the parent node
	  }
	else if (table[parent].right == index)
	  {
	    // we are the right child; we include the parent node plus the
	    //   left subtree
	    rank = rank + table[table[parent].left].rank +
	      table[parent].item.third;
	  }
	else
	  abort();
	index = parent;
      }
    return(rank);
  }

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::iterator
  _PMAP_TEMPLATE::index_find (const Index& x) const
  {
    size_type t, u;
    Index rank(0);
  
    t = root;
    u = 0;
    for (;;)
      {
	if (t == 0 || (rank + table[t].rank) < x)
	  {
	    if (u != 0)
	      splay(u);
	    return(iterator(this, u));
	  }
	rank = rank + table[table[t].left].rank;
	if (x < rank)
	  {
	    rank = rank - table[table[t].left].rank;
	    t = table[t].left;
	  }
	else if (rank < x)
	  {
	    rank = rank + table[t].item.third;
	    t = table[t].right;
	  }
	else
	  {
	    u = t;
	    rank = rank - table[table[t].left].rank;
	    t = table[t].left;
	  }
      }
  }

#endif

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::iterator
  _PMAP_TEMPLATE::find (const Key& k) const
  {	
    // look up key
    size_type hv = hashFunctor(k) % hashTable.size();
    size_type index = hashTable[hv];
    while (index != 0)
      {
	if (table[index].item.first == k)
	  return(iterator(this, index));
	index = table[index].next;
      }
    return(iterator(this, 0));
  }	

#ifdef _PMAP_INDEXED

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::iterator
  _PMAP_TEMPLATE::index_lower_bound (const Index& x) const
  {
    size_type t, u;
    Index rank(0);
  
    t = root;
    u = 0;
    for (;;)
      {
	if (t == 0 || (rank + table[t].rank) < x)
	  {
	    if (u != 0)
	      splay(u);
	    return(iterator(this, u));
	  }
	rank = rank + table[table[t].left].rank;
	if (rank < x)
	  {
	    rank = rank + table[t].item.third;
	    t = table[t].right;
	  }
	else
	  {
	    u = t;
	    rank = rank - table[table[t].left].rank;
	    t = table[t].left;
	  }
      }
  }

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::iterator
  _PMAP_TEMPLATE::index_upper_bound (const Index& x) const
  {
    size_type t, u;
    Index rank(0);
  
    t = root;
    u = 0;
    for (;;)
      {
	if (t == 0 || (rank + table[t].rank) < x)
	  {
	    if (u != 0)
	      splay(u);
	    return(iterator(this, u));
	  }
	rank = rank + table[table[t].left].rank;
	if (x < rank)
	  {
	    u = t;
	    rank = rank - table[table[t].left].rank;
	    t = table[t].left;
	  }
	else
	  {
	    rank = rank + table[t].item.third;
	    t = table[t].right;
	  }
      }
  }

#endif

  _PMAP_METHOD
  void _PMAP_TEMPLATE::enlarge ()
  {
    static const unsigned long long primes[65] = {0ULL,
						  1ULL,
						  3ULL,
						  7ULL,
						  13ULL,
						  31ULL,
						  61ULL,
						  127ULL,
						  251ULL,
						  509ULL,
						  1021ULL,
						  2039ULL,
						  4093ULL,
						  8191ULL,
						  16381ULL,
						  32749ULL,
						  65521ULL,
						  131071ULL,
						  262139ULL,
						  524287ULL,
						  1048573ULL,
						  2097143ULL,
						  4194301ULL,
						  8388593ULL,
						  16777213ULL,
						  33554393ULL,
						  67108859ULL,
						  134217689ULL,
						  268435399ULL,
						  536870909ULL,
						  1073741789ULL,
						  2147483647ULL,
						  4294967291ULL,
						  8589934583ULL,
						  17179869143ULL,
						  34359738337ULL,
						  68719476731ULL,
						  137438953447ULL,
						  274877906899ULL,
						  549755813881ULL,
						  1099511627689ULL,
						  2199023255531ULL,
						  4398046511093ULL,
						  8796093022151ULL,
						  17592186044399ULL,
						  35184372088777ULL,
						  70368744177643ULL,
						  140737488355213ULL,
						  281474976710597ULL,
						  562949953421231ULL,
						  1125899906842597ULL,
						  2251799813685119ULL,
						  4503599627370449ULL,
						  9007199254740881ULL,
						  18014398509481951ULL,
						  36028797018963913ULL,
						  72057594037927931ULL,
						  144115188075855859ULL,
						  288230376151711717ULL,
						  576460752303423433ULL,
						  1152921504606846883ULL,
						  2305843009213693951ULL,
						  4611686018427387847ULL,
						  9223372036854775783ULL,
						  18446744073709551557ULL};

    if (++logSize > 8*sizeof(size_type))
      {
	--logSize;
	if (sizeof(size_type) == 2)
	  _PMAP_ERROR("cannot hold more than 2^16-1 items");
	else if (sizeof(size_type) == 4)
	  _PMAP_ERROR("cannot hold more than 2^32-1 items");
	else
	  _PMAP_ERROR("cannot hold any more items");
      }

    // expand tables
    size_type oldSize = table.size();
    hashTable.resize(primes[logSize]);
    table.resize(1ULL << logSize);
  
    // add new elements to the free list
    for (size_type i = oldSize; i < table.size(); ++i)
      table[i].next = i+1;
    table[table.size() - 1].next = freeList;
    freeList = oldSize;
  
    // reconstruct the hash table
    for (size_type i = 0; i < hashTable.size(); ++i)
      hashTable[i] = 0;
    size_type hv;
    for (size_type i = 1; i < oldSize; ++i)
      if (table[i].parent != 0 || i == root)
	{
	  hv = hashFunctor(table[i].item.first) % hashTable.size();
	  table[i].next = hashTable[hv];
	  hashTable[hv] = i;
#ifdef _PMAP_MULTI
	  table[i].prev = 0;
	  table[table[i].next].prev = i;
#endif
	}
  }

  _PMAP_METHOD
  void _PMAP_TEMPLATE::splay (size_type index)
  {
    size_type l, r;
    size_type t;
    size_type y;
#ifdef _PMAP_INDEXED
    Index l_rank, r_rank;
#endif
  
#ifdef _PMAP_DEBUG
    if (root != 0 && table[root].parent != 0)
      _PMAP_ERROR("corrupt splay tree");
#endif
  
    if (root == 0)
      return;
  
    table[0].left = table[0].right = 0;
    l = r = 0;
#ifdef _PMAP_INDEXED
    l_rank = r_rank = 0;
#endif
  
    t = root;
    for (;;)
      {
	if (t == index)
	  break;
	if (lessFunctor(table[index].item.second, table[t].item.second) ||
	    !lessFunctor(table[t].item.second, table[index].item.second) &&
	    index < t)
	  {
	    if (table[t].left == 0)
	      break;
	    if (lessFunctor(table[index].item.second,
			    table[table[t].left].item.second) ||
		!lessFunctor(table[table[t].left].item.second,
			     table[index].item.second) &&
		index < table[t].left)
	      {
		// rotate right
		y = table[t].left;
		table[t].left = table[y].right;
		table[table[t].left].parent = t;
		table[y].right = t;
		table[t].parent = y;
#ifdef _PMAP_INDEXED
		table[t].rank = table[table[t].left].rank +
		  table[table[t].right].rank + table[t].item.third;
#endif
		t = y;
		if (table[t].left == 0)
		  break;
	      }
	    // link right
	    table[r].left = t;
	    table[t].parent = r;
	    r = t;
	    t = table[t].left;
#ifdef _PMAP_INDEXED
	    r_rank = r_rank + table[r].item.third + table[table[r].right].rank;
#endif
	  }
	else
	  {
	    if (table[t].right == 0)
	      break;
	    if (lessFunctor(table[table[t].right].item.second,
			    table[index].item.second) ||
		!lessFunctor(table[index].item.second,
			     table[table[t].right].item.second) &&
		table[t].right < index)
	      {
		// rotate left
		y = table[t].right;
		table[t].right = table[y].left;
		table[table[t].right].parent = t;
		table[y].left = t;
		table[t].parent = y;
#ifdef _PMAP_INDEXED
		table[t].rank = table[table[t].left].rank +
		  table[table[t].right].rank + table[t].item.third;
#endif
		t = y;
		if (table[t].right == 0)
		  break;
	      }
	    // link left
	    table[l].right = t;
	    table[t].parent = l;
	    l = t;
	    t = table[t].right;
#ifdef _PMAP_INDEXED
	    l_rank = l_rank + table[l].item.third + table[table[l].left].rank;
#endif
	  }
      }
#ifdef _PMAP_INDEXED
    // make l_rank and r_rank the ranks of the left and right
    //   trees we just built
    l_rank = l_rank + table[table[t].left].rank;
    r_rank = r_rank + table[table[t].right].rank;
    table[t].rank = l_rank + r_rank + table[t].item.third;
#endif
  
    table[l].right = table[r].left = 0;
  
#ifdef _PMAP_INDEXED
    // The following two loops correct the rank fields of the right path
    //   from the left child of the root and the right path from the left
    //   child of the root.
    for (y = table[0].right; y != 0; y = table[y].right)
      {
	table[y].rank = l_rank;
	l_rank = l_rank - (table[y].item.third + table[table[y].left].rank);
      }
    for (y = table[0].left; y != 0; y = table[y].left)
      {
	table[y].rank = r_rank;
	r_rank = r_rank - (table[y].item.third + table[table[y].right].rank);
      }
#endif
  
    table[l].right = table[t].left;
    table[table[l].right].parent = l;
    table[r].left = table[t].right;
    table[table[r].left].parent = r;
    table[t].left = table[0].right;
    table[table[t].left].parent = t;
    table[t].right = table[0].left;
    table[table[t].right].parent = t;
    root = t;
    table[root].parent = 0;
  }

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::size_type _PMAP_TEMPLATE::max_size () const
  {
    if (sizeof(size_type) == 2)
      return((size_type) 0xffffU);
    if (sizeof(size_type) == 4)
      return((size_type) 0xffffffffUL);
    else if (sizeof(size_type) == 8)
      return((size_type) 0xffffffffffffffffULL);
    else
      _PMAP_ERROR("unexpected size_type length");
  }

  _PMAP_METHOD
  void _PMAP_TEMPLATE::clear ()
  {
    table.clear();
    hashTable.clear();
    logSize = 0;
    root = 0;
    nElements = 0;
    freeList = 0;
    Entry e;
#ifdef _PMAP_INDEXED
    e.item.third = Index(0);
    e.rank = e.item.third;
#endif
    table.push_back(e);
    hashTable.resize(1);
  }

  _PMAP_METHOD
  void _PMAP_TEMPLATE::resize(size_type hint)
  {
    while (hint > table.size())
      enlarge();
  }

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::size_type _PMAP_TEMPLATE::max_bucket_count() const
  {
    if (sizeof(size_type) == 2)
      return((size_type) 65521U);
    else if (sizeof(size_type) == 4)
      return((size_type) 4294967291UL);
    else if (sizeof(size_type) == 8)
      return((size_type) 18446744073709551557ULL);
    else
      _PMAP_ERROR("unexpected size_type length");
  }

  _PMAP_METHOD
  typename _PMAP_TEMPLATE::size_type _PMAP_TEMPLATE::elems_in_bucket(size_type n) const
  {
    size_type count = 0;
    size_type index = hashTable[n];
    while (index != 0)
      {
	++count;
	index = table[index].next;
      }
    return(count);
  }

  _PMAP_METHOD
  void _PMAP_TEMPLATE::check () const
  {
    if ((1 << logSize) != table.size())
      _PMAP_ERROR("table.size() != 2^logSize");
    if (nElements >= table.size())
      _PMAP_ERROR("nElements invalid");
    size_type hashCount = 0;
    size_type e;
#ifdef _PMAP_MULTI
    size_type pe;
#endif
    size_type hv;
    for (size_type i = 0; i < hashTable.size(); ++i)
      {
#ifdef _PMAP_MULTI
	pe = 0;
#endif
	for (e = hashTable[i]; e != 0; e = table[e].next)
	  {
	    if (++hashCount > nElements)
	      _PMAP_ERROR("loop in hash table list");
	    hv = hashFunctor(table[e].item.first) % hashTable.size();
	    if (hv != i)
	      _PMAP_ERROR("hash table entry in wrong list");
#ifdef _PMAP_MULTI
	    if (table[e].prev != pe)
	      _PMAP_ERROR("prev link wrong in hash bucket");
	    pe = e;
#endif
	  }
      }
    if (hashCount != nElements)
      _PMAP_ERROR("number of elements in hash table incorrect");

    size_type freeCount = 0;
    for (e = freeList; e != 0; e = table[e].next)
      if (++freeCount >= table.size())
	_PMAP_ERROR("loop in free list");
    if (freeCount + nElements < table.size() - 1)
      _PMAP_ERROR("incomplete free list");
    if (freeCount + nElements > table.size() - 1)
      _PMAP_ERROR("free list includes non-empty entries");

    if (root >= table.size())
      _PMAP_ERROR("root invalid");
    if (root != 0)
      checkSubtree(root, 0,
		   false, table[root].item.second,
		   false, table[root].item.second);
  }

  _PMAP_METHOD
  void _PMAP_TEMPLATE::checkSubtree (size_type subtree, size_type parent,
				     bool checkMin, Value minValue,
				     bool checkMax, Value maxValue) const
  {
    if (table[subtree].parent != parent)
      _PMAP_ERROR("parent link invalid");
    if (checkMin && lessFunctor(table[subtree].item.second, minValue))
      _PMAP_ERROR("entry value out-of-order (min check)");
    if (checkMax && lessFunctor(maxValue, table[subtree].item.second))
      _PMAP_ERROR("entry value out-of-order (max check)");
    size_type left = table[subtree].left;
#ifdef _PMAP_INDEXED
    Index rank = 0;
#endif
    if (left != 0)
      {
#ifdef _PMAP_INDEXED
	rank = rank + table[left].rank;
#endif
	checkSubtree(left, subtree,
		     checkMin, minValue,
		     true, table[subtree].item.second);
      }
    size_type right = table[subtree].right;
    if (right != 0)
      {
#ifdef _PMAP_INDEXED
	rank = rank + table[right].rank;
#endif
	checkSubtree(right, subtree,
		     true, table[subtree].item.second,
		     checkMax, maxValue);
      }
#ifdef _PMAP_INDEXED
    rank = rank + table[subtree].item.third;
    if (table[subtree].rank != rank)
      _PMAP_ERROR("incorrect rank stored for table entry");
#endif
  }

  _PMAP_METHOD
  void _PMAP_TEMPLATE::checkNotInTree (size_type subtree, size_type index) const
  {
    if (subtree == index)
      _PMAP_ERROR("index should not be in tree");
    size_type left = table[subtree].left;
    if (left != 0)
      checkNotInTree(left, index);
    size_type right = table[subtree].right;
    if (right != 0)
      checkNotInTree(right, index);
  }

}
