//
//      pmap_common.h - common header for priority_map, priority_multimap,
//                      indexed_priority_map, indexed_priority_multimap
//
//      VERSION 1.0.0
//
//      Copyright (c) 2003,2005,2006,2007,2014 Pittsburgh Supercomputing Center
//
//      This program is free software; you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation; either version 2 of the License, or
//      (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.  Neither the
//      Pittsburgh Supercomputing Center nor any of the authors assume
//      any liability for damages, incidental or otherwise, caused by the
//      installation or use of this software.
//
//      You should have received a copy of the GNU General Public License
//      along with this program; if not, write to the Free Software
//      Foundation, Inc., 51 Franklin Street, Fifth Floor,
//      Boston, MA  02110-1301  USA
//
//
//      This implementation is based on the splay tree code
//      (top-down-size-splay.c) written by Daniel Sleator, and which he
//      has released into the public domain.
//
//	HISTORY
//      	3/03	Written by Greg Hood, PSC
//              3/05    Removed NEOSIM dependencies (ghood@psc.edu)
//              10/05   Extended from indexed_priority_map (ghood@psc.edu)
//              1/07    Changed string to std::string (ghood@psc.edu)
//              3/14    Converted to use unordered_map instead of hash_map
//

#undef _PMAP_TEMPLATE
#undef _PMAP_ITEM
#ifdef _PMAP_INDEXED
  #define _PMAP_TEMPLATE	_PMAP_CLASS<Key,Value,Index,H,L,A>
  #define _PMAP_ITEM		triple<Key,Value,Index>
#else
  #define _PMAP_TEMPLATE	_PMAP_CLASS<Key,Value,H,L,A>
  #define _PMAP_ITEM		pair<Key,Value>
#endif

namespace pmap
{
  using std::size_t;
  using std::pair;
  using std::ptrdiff_t;
  using std::vector;

  template<class Key,
	   class Value,
#ifdef _PMAP_INDEXED
           class Index = size_t,  // must support operators +,-,<,=,Index(int)
#endif
           class H = std::hash<Key>,
           class L = std::less<Value>,
           class A = std::allocator< _PMAP_ITEM >
          >
    class _PMAP_CLASS
    {
      public:
      explicit _PMAP_CLASS (const H& h = H(),
			    const L& l = L(),
			    const A& a = A());
      _PMAP_CLASS (const _PMAP_CLASS& x);
      _PMAP_CLASS& operator= (const _PMAP_CLASS& x);
      void swap (_PMAP_CLASS& x);

      typedef typename A::pointer pointer;
      typedef typename A::const_pointer const_pointer;
      typedef typename A::reference reference;
      typedef typename A::const_reference const_reference;

#ifdef _PMAP_SIZE_TYPE
      typedef _PMAP_SIZE_TYPE size_type;
#else
      typedef size_t size_type;
#endif
#ifdef _PMAP_DIFFERENCE_TYPE
      typedef _PMAP_DIFFERENCE_TYPE difference_type;
#else
      typedef ptrdiff_t difference_type;
#endif

      struct error: public std::exception
      {
	error(const std::string& s) { description = s; }
	~error() throw() {}
	const char* what() { return(description.c_str()); }
	std::string description;
      };

      class iterator :
#ifdef _PMAP_INDEXED
      public std::iterator<std::random_access_iterator_tag,
                           _PMAP_ITEM, 
                           Index,
                           const _PMAP_ITEM*,
                           const _PMAP_ITEM&>
#else
      public std::iterator<std::bidirectional_iterator_tag,
                           _PMAP_ITEM,
                           ptrdiff_t,
                           const _PMAP_ITEM*,
                           const _PMAP_ITEM&>
#endif
      {
      public:
	iterator ()
	{
	  this->ipm = 0;
	  index = 0;
	}
	iterator& operator= (const iterator& x)
	{
	  this->ipm = x.ipm;
	  this->index = x.index;
	  return(*this);
	}
	const _PMAP_ITEM& operator* () const
	{
	  if (index == 0)
	    abort();
	  else
	    return(ipm->table[index].item);
	}
	const _PMAP_ITEM* operator-> () const
	{
	  if (index == 0)
	    return(0);
	  else
	    return(&(ipm->table[index].item));
	} 
	iterator& operator++ ();
	iterator operator++ (int);
	iterator& operator-- ();
	iterator operator-- (int);
	bool operator== (const iterator& x) const
	{
	  if (ipm != x.ipm)
	    return(false);
	  return(index == x.index);
	}
	bool operator!= (const iterator& x) const
	{
	  if (ipm != x.ipm)
	    return(true);
	  return(index != x.index);
	}
	bool operator< (const iterator& x) const;
	bool operator> (const iterator& x) const;
	bool operator<= (const iterator& x) const;
	bool operator>= (const iterator& x) const;
#ifdef _PMAP_INDEXED
	iterator operator+ (const Index x);
	iterator& operator+= (const Index x);
	iterator operator- (const Index x);
	iterator& operator-= (const Index x);
	const _PMAP_ITEM& operator[] (const Index x);
#endif
      private:
	iterator (const _PMAP_TEMPLATE* ipm,
			size_type i = 0)
	{
	  this->ipm = ipm;
	  index = i;
	}

	const _PMAP_TEMPLATE* ipm;
	size_type index;
	friend class _PMAP_CLASS;
      };

      typedef iterator const_iterator;
      typedef typename std::reverse_iterator<iterator> reverse_iterator;
      typedef reverse_iterator const_reverse_iterator;

      size_type erase (const Key& k);         // erases all entries
                                              //   having the given key
      void erase (iterator& pos);             // erases the designated
                                              //   entry and updates the
                                              //   iterator to point to
                                              //   the next entry
      void erase (iterator& first,
		  iterator& last);

      iterator front () const
      {
	return(begin());
      };
      iterator back () const
      {
	iterator i = end();
	return(--i);
      }

      iterator top () const                 // returns the entry with the
      {                                           //   highest value
	size_type x = root;
	if (x != 0)
	  while (table[x].right != 0)
	    x = table[x].right;
	return(iterator(this, x));
      }                                    
      void pop ()                               // removes the highest-valued
      { iterator i = top(); erase(i); }   //   entry

      iterator find (const Key& k) const; // returns an entry with the
                                                //   given key
      iterator lower_bound (const Value& v) const;
                                                //  returns the first entry
                                                //    having a value >=
                                                //    the given value
      iterator upper_bound (const Value& v) const;
                                                //  returns the first entry
                                                //    having a value >
                                                //    the given value
      size_type count (const Key& k) const;  // returns the number of elements
                                             //   with the given key

      iterator begin() const
      {
	size_type x = root;
	if (x != 0)
	  while (table[x].left != 0)
	    x = table[x].left;
	return(iterator(this, x));
      }
      iterator end () { return(iterator(this, 0)); }
      reverse_iterator rbegin () { return(reverse_iterator(end())); }
      reverse_iterator rend () const { return(reverse_iterator(begin())); }
      bool empty () const { return(nElements == 0); }
      size_type size () const { return(nElements); }
      size_type max_size () const;
      void clear ();
      void resize (size_type hint);
      size_type bucket (const Key& k) const { return(hashFunctor(k) %
						     hashTable.size()); }
      size_type bucket_count() const { return(hashTable.size()); }
      size_type max_bucket_count() const;
      size_type elems_in_bucket(size_type n) const;
      void check () const;
  
#ifdef _PMAP_INDEXED

#ifdef _PMAP_MULTI
      iterator
#else
      pair<iterator,bool>
#endif
      insert (const pair<Key,Value>& x)
      {
	return(insert(x.first, x.second, Index(1)));
      }

#ifdef _PMAP_MULTI
      iterator
#else
      pair<iterator,bool>
#endif
      insert (const Key& k, const Value& v)
      {
	return(insert(k, v, Index(1)));
      }

#ifdef _PMAP_MULTI
      iterator
#else
      pair<iterator,bool>
#endif
      insert (const triple<Key,Value,Index>& x)
      {
	return(insert(x.first, x.second, x.third));
      }

#ifdef _PMAP_MULTI
      iterator
#else
      pair<iterator,bool>
#endif
      insert (const Key& k, const Value& v, const Index& delta);

      void push (const triple<Key,Value,Index>& x)
      {
	(void) insert(x.first, x.second, x.third);
      }
      void push (const Key& k, const Value& v, const Index& delta)
      {
	(void) insert(k, v, delta);
      }
      void push (const pair<Key,Value>& x)
      {
	(void) insert(x.first, x.second, Index(1));
      }
      void push (const Key& k, const Value& v)
      {
	(void) insert(k, v, Index(1));
      }                                       
      Index index (const Key& k) const;       // returns the index of the
                                              //   lowest-valued entry
                                              //   with the given key
      Index index (const iterator& i) const; // returns the index of the
                                             //   designated entry
      iterator index_find (const Index& x) const;
                                                //  returns the entry with
                                                //    exactly the given index,
                                                //    or end() if none has
      iterator index_lower_bound (const Index& x) const;
                                                //  returns the first entry
                                                //    having an index >=
                                                //    the given index
      iterator index_upper_bound (const Index& x) const;
                                                //  returns the first entry
                                                //    having an index >
                                                //    the given index
#else  // !_PMAP_INDEXED

#ifdef _PMAP_MULTI
      iterator
#else
      pair<iterator,bool>
#endif
      insert (const pair<Key,Value>& x)
      {
	insert(x.first, x.second);
      }

#ifdef _PMAP_MULTI
      iterator
#else
      pair<iterator,bool>
#endif
      insert (const Key& k, const Value& v);

      void push (const pair<Key,Value>& x)
      {
	(void) insert(x.first, x.second);
      }

      void push (const Key& k, const Value& v)
      {
	(void) insert(k, v);
      }
#endif

    private:
      void insertInTree (size_type index);
      void removeFromTree (size_type index);

      struct Entry
      {
	Entry ()
	{
	  left = 0; right = 0; parent = 0; next = 0;
#ifdef _PMAP_MULTI
	  prev = 0;
#endif
	}
	size_type left;
	size_type right;
	size_type parent;
	size_type next;
#ifdef _PMAP_MULTI
	size_type prev;
#endif
#ifdef _PMAP_INDEXED
	Index rank;
	triple<Key,Value,Index> item;
#else
	pair<Key,Value> item;
#endif
      };

      H hashFunctor;
      L lessFunctor;
      typedef typename A::template rebind<size_type>::other HashAlloc;
      typedef typename A::template rebind<Entry>::other EntryAlloc;

      size_type logSize;       // log2 of the size of the table
      size_type freeList;      // points to the first entry on the free list
                                  //    for the entry table (a value of 0
                                  //    indicates an empty free list); the
                                  //    next field is used to chain together
                                  //    the free list
      size_type nElements;     // number of elements in the tree
      size_type root;          // root of the splay tree (0 if empty)
      vector<size_type, HashAlloc> hashTable;
                                  // each entry is the head of a hash
                                  //   list for the corresponding bucket
                                  //   (a 0 value indicates a null list)
      vector<Entry, EntryAlloc> table;
                                  // holds all the entries in the tree;
                                  //    the 0th entry is special and must
                                  //    remain unused

      void enlarge ();
      void splay (size_type index);
      void checkSubtree (size_type subtree, size_type parent,
			 bool checkMin, Value minValue,
		     bool checkMax, Value maxValue) const;
      void checkNotInTree (size_type subtree, size_type index) const;
    };

}
